package ihm;
import calc.Maneuver;
import fr.cnes.genius.exception.GException;
import fr.cnes.genius.exception.GIntervalException;
import fr.cnes.genius.highLevel.GEntryReal;
import fr.cnes.genius.interval.GInterval;
import fr.cnes.genius.lowLevel.GContainer;
import fr.cnes.genius.main.GClear;
import fr.cnes.genius.main.GDisplay;
import fr.cnes.genius.main.GReadWrite;
import fr.cnes.genius.unit.GMetricUnit;
import fr.cnes.genius.unit.GUnit;
import fr.cnes.genius.utils.GPanTest;

/**
 * Class defining the maneuver widget
 * @author goesterjf
 *
 */
public class GManeuver extends GContainer implements GDisplay, GReadWrite, GClear {
		
	/** Maneuver duration widget */
	private GEntryReal durationIhm;
	/** Maneuver thrust level widget */
	private GEntryReal thrustIhm;
	/** Maneuver specific impulse widget */
	private GEntryReal ispIhm;
	
	// Units
	private GUnit[] unitDuration = {new GMetricUnit("mn"), new GMetricUnit("s")};
	private GUnit[] unitThrust = {new GMetricUnit("N")};
	private GUnit[] unitIsp = {new GMetricUnit("s")};
	
	// Validity intervals
	public final GInterval intErrDoublePositive = new GInterval(0., Double.POSITIVE_INFINITY, GInterval.Rule.INCLUSIVE, GInterval.Rule.EXCLUSIVE);
	public final GInterval intErrWarIsp = new GInterval(250., 350., GInterval.Rule.INCLUSIVE, GInterval.Rule.EXCLUSIVE, 200., 400., GInterval.Rule.INCLUSIVE, GInterval.Rule.EXCLUSIVE);

	/**
	 * Constructor with by default values.
	 * @throws GIntervalException 
	 */
	public GManeuver ( ) throws GIntervalException {
		
		this(new Maneuver(0., 0., 0.));

	}

	/**
	 * Constructor with a Maneuver Object as input data.
	 * @param man	maneuver object
	 * @throws GIntervalException 
	 */
	public GManeuver ( Maneuver man ) throws GIntervalException {

		durationIhm =  new GEntryReal("Duration:", man.getDuration(), unitDuration);
		durationIhm.setNameInConfigFile("duration");
		durationIhm.addGInterval(intErrDoublePositive);

		thrustIhm =  new GEntryReal("Thrust:", man.getThrust(), unitThrust);
		thrustIhm.setNameInConfigFile("thrust");
		thrustIhm.addGInterval(intErrDoublePositive);

		ispIhm = new GEntryReal("Isp:", man.getIsp(), unitIsp);
		ispIhm.setNameInConfigFile("isp");
		ispIhm.addGInterval(intErrWarIsp);

	}
	
	/**
	 * Generic method override.
	 */
	public void generic() throws GException {
		put(durationIhm);
		put(thrustIhm);
		put(ispIhm);
	}

	/**
	 * Display method : only use generic one.
	 */
	public void display() throws GException { generic(); }

	/**
	 * Read method : only use generic one.
	 */
	public void read() throws GException { generic(); }

	/**
	 * Write method : only use generic one.
	 */
	public void write() throws GException { generic(); }

	/**
	 * Clear method : only use generic one.
	 */
	public void clear() throws GException { generic(); }

	/**
	 * Method giving a Maneuver object.
	 * @return	Maneuver object
	 */
	public Maneuver getManeuver () {
		return new Maneuver(durationIhm.getValue(), thrustIhm.getValue(), ispIhm.getValue());
	}
	
	/**
	 * Executable method to test the widget unitarily
	 * @param args no args ...
	 * @throws GIntervalException 
	 */
	public static void main(String[] args) throws GIntervalException {
		
		GManeuver man = new GManeuver(new Maneuver(0., 0., 0.));
		GPanTest testMan = new GPanTest("Test GManoeuvre", man, "man.xml");
		testMan.display();
		
	}

}
