

import java.awt.Color;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import fr.cnes.genius.chart.GCoordinatesData;
import fr.cnes.genius.exception.GException;
import fr.cnes.genius.exception.GPlotDataReaderException;
import fr.cnes.genius.lowLevel.GFrame;
import fr.cnes.genius.lowLevel.GPanel;
import fr.cnes.genius.plots.GGroundPlotPanel;
import fr.cnes.genius.plots.GPlotColumnInfo;
import fr.cnes.genius.plots.GPlotDataMadonaReader;
import fr.cnes.genius.plots.GPlotDataReaderInterface;
import fr.cnes.genius.plots.GroundTrackDataProvider;
import fr.cnes.genius.sqlite.GPlotDataSqliteReader;

/**
 * Tutorial for using GGroundPlotPanel.
 * @author goesterjf
 *
 */
public class TestForPlanisphere extends GPanel {
    
    /** 90 degrees (used for initial zoom and gap threshold */
    protected static final double DEG90 = 90.;
    /** 180 degrees (used for initial zoom and gap threshold */
    private static final double DEG180 = 180.;

    private GPlotDataReaderInterface reader = null;

    private GGroundPlotPanel plot;
    private String nameLon = "LONG";
    private String nameLat = "LATG";
        
    public TestForPlanisphere() {
        
        GroundTrackDataProvider prov = new GroundTrackDataProvider() {
            
            @Override
            public List<GCoordinatesData> getData(String fileName) {
              final ArrayList<GCoordinatesData> listOfData = new ArrayList<GCoordinatesData>();
              
              // Open, read and plot trajectory
              openDbFile(fileName);
              ArrayList<Double[]> data1 = scanFile(0, 0.);
              ArrayList<Double[]> data2 = scanFile(0, 90.);
              listOfData.add(new GCoordinatesData("Nominal", data1, Color.RED, null, DEG180, DEG90));
              listOfData.add(new GCoordinatesData("Shifted", data2, Color.GREEN, null, DEG180, DEG90));

              return listOfData;
            }
        };
        
        plot = new GGroundPlotPanel(prov);
        
    }
    
    /**
     * Open file either ASCII file or sqLite file.
     * @param fileName  File name
     */
    private void openDbFile ( final String fileName ) {
        
        final File file = new File(fileName);
        if ( file.getName().endsWith("db") ) {
            reader = new GPlotDataSqliteReader();
        } else {
            reader = new GPlotDataMadonaReader();
        }
        try {
            reader.load(file);
        } catch (GPlotDataReaderException err) {
            err.printStackTrace();
        }
        
    }

    /**
     * Method allowing to read a column file and to extract the data to be plotted.
     * @param dataSetNum  Number of dataSet
     * @param dataSetNum  Longitude shift
     * @return            data to be plotted
     */
    private ArrayList<Double[]> scanFile ( final int dataSetNum, final double deltaLon ) {
        
        List<String> dataSetList = null;
        String dataSetId = null;
        try {
            dataSetList = reader.getDatasetList();
            if ( dataSetList != null ) {
                dataSetId = dataSetList.get(dataSetNum);
            }
        } catch (GPlotDataReaderException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        
        // We get the rank of both columms
        final int nbCol = reader.getNumberColums(dataSetId);
        int numColAbs = -1;
        int numColOrd = -1;
        for (int i = 0; i < nbCol; i++) {
            final GPlotColumnInfo infos = reader.getColumnInfo(dataSetId, i);
            if ( infos.getName().equals(nameLon) ) {
                numColAbs = infos.getIndex();
            } else if ( infos.getName().equals(nameLat) ) {
                numColOrd = infos.getIndex();
            }
        }
        
        // We store data to plot
        return storeData(reader, dataSetId, numColAbs, numColOrd, deltaLon);
    
    }
    
    /**
     * Method reading abscissa and ordinate coordinates to be plotted.
     * @param reader      reader where to get data
     * @param dataSetId   Name of dataSet (may be null)
     * @param numColAbs   index of the abscissa column
     * @param numColOrd   index of the ordinate column
     * @param dataSetNum  Longitude shift
     * @return            data to be plotted
     */
    private ArrayList<Double[]> storeData (final GPlotDataReaderInterface reader,
            final String dataSetId,
            final int numColAbs, final int numColOrd, final double deltaLon ) {
        
        Locale.setDefault(Locale.US);
        List<Double[]> data = null;
        if ( (numColAbs >= 0) && (numColOrd >= 0 )) {
            try {
                data = reader.getXYColumns(dataSetId, numColAbs, numColOrd);
            } catch (GPlotDataReaderException err) {
                err.printStackTrace();
            }
        }
        
        // We set X data between -180 deg and +180deg
        for (int j = 0; j < data.get(0).length; j++) {
                final double lon = data.get(0)[j] + deltaLon;
                data.get(0)[j] = lon;
            if ( data.get(0)[j] > DEG180 ) {
                data.get(0)[j] = data.get(0)[j] - 2*DEG180;
            } else if ( data.get(0)[j] < -DEG180 ) {
                data.get(0)[j] = data.get(0)[j] + 2*DEG180;                    
            }
        }
        
        return (ArrayList<Double[]>)data;

    }

    @Override
    public void display() throws GException {
        generic();
    }

    @Override
    public void generic() throws GException {
        put(plot);
    }

    public static void main(String[] args) {
        TestForPlanisphere pan = new TestForPlanisphere();
        GFrame frame = new GFrame("TestForPlanisphere", pan); 
        frame.display();
    }

}
