import java.awt.Dimension;
import java.awt.Shape;
import java.io.File;
import java.util.ArrayList;
import java.util.List;

import fr.cnes.genius.chart.GShapeFactory;
import fr.cnes.genius.exception.GException;
import fr.cnes.genius.exception.GPlotDataReaderException;
import fr.cnes.genius.exception.SqliteException;
import fr.cnes.genius.lowLevel.GFrame;
import fr.cnes.genius.plots.GPlotColumnInfo;
import fr.cnes.genius.plots.GPlotDataMadonaReader;
import fr.cnes.genius.plots.GPlotPanel;
import fr.cnes.genius.sqlite.ColumnInfo.ColumnType;
import fr.cnes.genius.sqlite.GPlotDataSqliteReader;
import fr.cnes.genius.sqlite.ResultWriter;

/**
 * Class to test MADONA files reader, SQLITE writer then GPlotpanel.
 * @author goesterjf
 *
 */
public class TestForMadonaAndSqlLite {
    
    private final String[] dataNames;
    
    private Integer[] numCols;
    private Double[]  gapThresholds;
    private String[]  unitNames;
    private String[]  description;
    
    private List<Double[]> dataEphem;
    private List<Double[]> dataEvent;
    
    private int nbColumnsMadona;
    private int nbColumnsSql;
    private int nbPtsEphem;
    private int nbPtsEvent;
    
    private static final int FREQ_EVENT = 100;
    
    private static final String NAME_EPHEM_TABLE = "ephemTable";
    private static final String NAME_EVENT_TABLE = "eventTable";

    /**
     * Constructor
     * @param dataNames                 List of variable to put in the sqLite data base.
     * @throws GPlotDataReaderException GENIUS exception
     * @throws SqliteException          Sqlite Exception
     */
    public TestForMadonaAndSqlLite( final String... dataNames ) throws GPlotDataReaderException, SqliteException {
        
        this.dataNames = dataNames;
        this.nbColumnsSql = dataNames.length;
        
    }
    
    /**
     * Reading the initial Madona file
     * @param madonaFileName            Name of the Madona file
     * @throws GPlotDataReaderException GENIUS exception
     * @throws SqliteException          Sqlite Exception
     */
    public void readMadonaFile ( final String madonaFileName ) throws GPlotDataReaderException, SqliteException {
        
        numCols = new Integer[dataNames.length];
        gapThresholds = new Double[dataNames.length];
        unitNames = new String[dataNames.length];
        description = new String[dataNames.length];
        
        // Reading the EPHEM.txt file
        
        // Opening it and load data
        final GPlotDataMadonaReader fileData = new GPlotDataMadonaReader();
        fileData.load(new File(madonaFileName));
        
        // Recovery of the colums information
        nbColumnsMadona = fileData.getNumberColums(null);
        
        // Loop on the columns to extract the aimed ones for sqLite data base
        for (int i = 0; i < nbColumnsMadona; i++) {
            
            final GPlotColumnInfo colInfo = fileData.getColumnInfo(null, i);

            for (int j = 0; j < dataNames.length; j++) {
                
                if ( dataNames[j].equals(colInfo.getName()) ) {
                    numCols[j] = colInfo.getIndex();
                    gapThresholds[j] = colInfo.getGapThreshold();
                    if ( dataNames[j].startsWith("LAT") || dataNames[j].startsWith("LON") ) {
                        gapThresholds[j] = 180.;
                    }
                    unitNames[j] = colInfo.getUnitName();
                    description[j] = colInfo.getDesc();
                }
                
            }
            
        }
        
        dataEphem = fileData.getColumns(null, numCols);
        nbPtsEphem = dataEphem.get(0).length;
        // Conversion in SI units
        for (int iCol = 0; iCol < nbColumnsSql; iCol++) {
            if ( unitNames[iCol].equals("km") ) {
                for (int iPts = 0; iPts < nbPtsEphem; iPts++) {
                    dataEphem.get(iCol)[iPts] = dataEphem.get(iCol)[iPts]*1000.;
                }
            } else if ( unitNames[iCol].equals("deg") ) {
                for (int iPts = 0; iPts < nbPtsEphem; iPts++) {
                    dataEphem.get(iCol)[iPts] = Math.toRadians(dataEphem.get(iCol)[iPts]);
                }
            }
            
        }
        
        // Loop to build events data (considering an event each 10 points of the ephemeris initial file)
        dataEvent = new ArrayList<Double[]>();
        if ( nbPtsEphem%FREQ_EVENT == 0 ) {
            nbPtsEvent = nbPtsEphem/FREQ_EVENT + 1;             
        } else {
            nbPtsEvent = nbPtsEphem/FREQ_EVENT + 2;
        }
        for (int iVar = 0; iVar < dataEphem.size(); iVar++) {
            // Intermediate list
            final Double[] listTmp = new Double[nbPtsEvent];
            int iEvent = 0;
            for (int iEphem = 0; iEphem < nbPtsEphem; iEphem++) {
                if ( (iEphem%FREQ_EVENT  == 0) || (iEphem == nbPtsEphem-1) ) {
                    listTmp[iEvent] = dataEphem.get(iVar)[iEphem];
                    iEvent = iEvent + 1;
                }
            }
            dataEvent.add(listTmp);
        }
        
    }
    
    /**
     * Building the sqLite data base.
     * @param sqlFileName       Name of the sqLite data base
     * @throws SqliteException  Sqlite Exception
     */
    public void writeSqlFile ( final String sqlFileName ) throws SqliteException {
        
        // File creation and reset if it already exists
        final File sqliteFile = new File(sqlFileName);
        if ( sqliteFile.exists() && !sqliteFile.delete() ) {
            System.out.println("Sqlite output file failed to delete: %s");
        }

        final ResultWriter  resultWriter = new ResultWriter(sqliteFile);
        resultWriter.open();
        
        // Table configuration
        resultWriter.addTable (NAME_EPHEM_TABLE);
        resultWriter.addTable (NAME_EVENT_TABLE);         
         
        // Columns configuration
        /// Note: 
        // gapThreshold is the value given for plot discontinuities
        // userVisible may be used to add columns but not visible via the GUI (for example for plotting).
        for (int i = 0; i < nbColumnsSql; i++) {
            resultWriter.addColumn(NAME_EPHEM_TABLE, dataNames[i], description[i], ColumnType.REAL, unitNames[i], gapThresholds[i], true);
            resultWriter.addColumn(NAME_EVENT_TABLE, dataNames[i], description[i], ColumnType.REAL, unitNames[i], gapThresholds[i], true);
        }
         
        // Adding values for each line
        // Note: each table is independant and one may fill them asynchroneously.
        for (int i = 0; i < nbPtsEphem; i++) {
            // Note: it is mandatory to add a value for each column. If it is not the case, it will raise an error.
            // On the other hand, if we add two times a value for the same line/column, the last value will be taken into account.
            for (int j = 0; j < nbColumnsSql; j++) {
                resultWriter.addValue(NAME_EPHEM_TABLE, dataNames[j], dataEphem.get(j)[i]);                          
            }
            // Line is stored in the database
            resultWriter.writeLine(NAME_EPHEM_TABLE);
        }
        // Same for events
        for (int i = 0; i < nbPtsEvent; i++) {
            for (int j = 0; j < nbColumnsSql; j++) {
                resultWriter.addValue(NAME_EVENT_TABLE, dataNames[j], dataEvent.get(j)[i]);                          
            }
            resultWriter.writeLine(NAME_EVENT_TABLE);
        }
         
        
        // Closing the data base
        resultWriter.close();

    }

    /**
     * @param args
     * @throws GException 
     * @throws SqliteException 
     */
    public static void main(String[] args) throws GException, SqliteException {
        
        TestForMadonaAndSqlLite test = new TestForMadonaAndSqlLite("TIME", "ALTG", "LATG", "LONG");
        test.readMadonaFile("EPHEM.txt");        
        test.writeSqlFile("EPHEM.db");
        
        final GPlotPanel plots = new GPlotPanel("EPHEM.db", "", "Ephemeris file", 0, false, new GPlotDataSqliteReader());
        plots.setSelectedFile(new File("EPHEM.db"));
        plots.setDatasetPlotProperties(NAME_EPHEM_TABLE, null, null);
        Shape shape = GShapeFactory.createRectangle(10, 10);
        plots.setDatasetPlotProperties(NAME_EVENT_TABLE, shape, null);

        final GFrame frame = new GFrame("TestForMadonaAndSqlLite", plots);    
        frame.setPreferredSize(new Dimension(800, 650));
        frame.display();

    }

}
